package gov.va.vamf.scheduling.facility.timezone;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.core.GenericType;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;
import java.net.URI;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

@Service
public class FacilityTimezoneService {

    private static final Log logger = LogFactory.getLog(FacilityTimezone.class);
    private static final String DEFAULT_TIMEZONE = "America/New_York";

    @Value("${facility.timezone.fetch.all.uri}")
    private String facilityTimezoneFetchAllUri;

    @Value("${facility.timezone.fetch.by.id.uri}")
    private String facilityTimezoneFetchByIdUri;

    public FacilityTimezone fetchFacilityTimezoneById(String facilityId) {

        Map<String, String> pathParams = new HashMap<>();
        pathParams.put("facility-id", "dfn-" + facilityId);

        URI uri = buildUri(facilityTimezoneFetchByIdUri, pathParams);

        String uriString = uri.toString();
        return getRestClient()
            .target(uriString)
            .request()
            .accept(MediaType.APPLICATION_JSON)
            .get(FacilityTimezone.class);
    }

    public Date getFacilityAdjustedDate(String facilityId) {
        final TimeZone tz = TimeZone.getTimeZone(getFacilityTimezone(facilityId));
        Date now = new Date();
        return new Date(now.getTime() + tz.getOffset(now.getTime()));
    }

    private Client getRestClient() {
        return ClientBuilder.newBuilder().build();
    }

    private URI buildUri(String url, Map<String, String> pathParams) {
        return UriBuilder.fromUri(url).buildFromMap(pathParams);
    }

    private String getFacilityTimezone(String facilityId) {
        FacilityTimezone facilityTimezone = fetchFacilityTimezoneById(facilityId);

        String tz;
        if (facilityTimezone != null && facilityTimezone.getTimezone() != null) {
            tz = facilityTimezone.getTimezone();
        } else {
            logger.warn("No timezone found for facility: " + facilityId);
            tz = DEFAULT_TIMEZONE;
        }
        return tz;
    }
}
